# OEM Integration Guide
**Product:** MetaSpace InductiveGuard™
**Target Audience:** Firmware Engineers (Embedded C++)
**Supported Platforms:** ARM Cortex-M, MSP430, RISC-V, AVR

---

## 1. System Requirements

### Hardware
*   **ADC Sampling Rate:** Minimum **10 kHz** (100µs period) is recommended for detecting micro-fractures before thermal runaway.
*   **Sensors:**
    *   Coil Voltage Monitor (Precision: +/- 10mV)
    *   Coil Current Monitor (Precision: +/- 1mA)

### Software
*   **Compiler:** C++11 compliant or later (GCC, Clang, IAR, Keil, MSVC).
*   **Memory Footprint:** < 1KB Flash, < 100 Bytes RAM.

## 2. Integration Steps

### Step 1: Include the Core IP
Copy `InductiveGuard.hpp` and `InductiveGuard.cpp` into your project's source tree.

### Step 2: Instantiation
Create a global or persistent instance of the guard class. It maintains internal state (previous impedance) to calculate derivatives.

```cpp
#include "InductiveGuard.hpp"

// Persistent instance
InductiveGuard g_safety_monitor;
```

### Step 3: The Control Loop
Call the `audit()` function inside your main high-frequency control loop or ADC Ready Interrupt (ISR).

**CRITICAL:** This function must be called **synchronously** with every new data sample.

```cpp
void ADC_IRQHandler() {
    // 1. Read Hardware Sensors
    int32_t v_meas = Hardware_ReadVoltage_mV();
    int32_t i_meas = Hardware_ReadCurrent_mA();

    // 2. Prepare Data Structure
    SensorData data;
    data.voltage_mV = v_meas;
    data.current_mA = i_meas;

    // 3. Perform MetaSpace Safety Check (Latency < 5µs)
    SafetyStatus status = g_safety_monitor.audit(data);

    // 4. Act on Result
    if (status == SafetyStatus::VIOLATION) {
        // --- EMERGENCY STOP ---
        // 1. Open H-Bridge / Disconnect Coil
        Hardware_DisablePower();
        
        // 2. Log Fault
        System_Log("InductiveGuard: Micro-Fracture Detected!");
        
        // 3. Enter Safe State
        System_EnterSafeMode();
    } else {
        // Continue normal charging regulation
        PID_Loop(v_meas, i_meas);
    }
}
```

## 3. Troubleshooting

### False Positives
If the system triggers `VIOLATION` during normal startup:
*   **Cause:** Inrush current or contact instability.
*   **Fix:** Ensure `audit()` is only called *after* the connection is stable (e.g., add a 100ms startup delay masking).

### "Zero-Division" Safety
The kernel automatically returns `SAFE` if current is 0mA to prevent divide-by-zero errors. Ensure your system handles the 0mA state correctly in the outer loop.
