import matplotlib
matplotlib.use('TkAgg')
import matplotlib.pyplot as plt
import matplotlib.animation as animation
import numpy as np
from datetime import datetime

class InductiveMonitor:
    def __init__(self):
        # --- Physics Constants ---
        self.current_target = 0.1      # 100mA constant current
        self.base_impedance = 2.0      # Ohms
        self.dt = 0.05                 # Time step (simulated seconds)
        self.heat_capacity = 0.5       # Arbitrary units for tissue volume
        self.cooling_factor = 0.05     # Passive cooling rate
        self.fracture_impedance = 15.0 # Ohms after fracture
        self.fracture_frame = 200      # Frame where fracture occurs
        
        # --- State Variables ---
        self.frame_count = 0
        self.time_data = []
        
        # Legacy System State
        self.legacy_temp = 37.0
        self.legacy_power_on = True
        self.legacy_temps = []
        
        # MetaSpace System State
        self.metaspace_temp = 37.0
        self.metaspace_power_on = True
        self.metaspace_temps = []
        
        # Shared Environment (Impedance)
        self.impedance = self.base_impedance
        self.impedance_data = []
        
    def update(self, frame):
        self.frame_count += 1
        current_time = self.frame_count * self.dt
        self.time_data.append(current_time)
        
        # --- 1. Physics Event: Micro-Fracture ---
        if self.frame_count >= self.fracture_frame:
            # Add some noise to the fracture
            noise = np.random.normal(0, 0.5)
            self.impedance = self.fracture_impedance + noise
        else:
            self.impedance = self.base_impedance + np.random.normal(0, 0.05)
            
        self.impedance_data.append(self.impedance)
        
        # --- 2. Legacy Logic (Reactive) ---
        # Power = I^2 * R
        if self.legacy_power_on:
            power_legacy = (self.current_target ** 2) * self.impedance
        else:
            power_legacy = 0
            
        # Thermal Model
        delta_temp_legacy = (power_legacy * self.dt / self.heat_capacity) - (self.legacy_temp - 37.0) * self.cooling_factor * self.dt
        self.legacy_temp += delta_temp_legacy
        
        # Legacy Safety Check: Thermistor > 43°C
        if self.legacy_temp > 43.0:
            self.legacy_power_on = False
            
        self.legacy_temps.append(self.legacy_temp)
        
        # --- 3. MetaSpace Logic (Vectorial SPS) ---
        # Calculate dR/dt
        dr_dt = 0
        if len(self.impedance_data) > 1:
            dr_dt = (self.impedance_data[-1] - self.impedance_data[-2]) / self.dt
            
        # MetaSpace Safety Check: dR/dt > 5 Ohm/ms (predictive)
        # Note: 5 Ohm/ms is huge, we'll check for 5 Ohm/s for this scale or just the instantaneous jump
        if dr_dt > 5.0: # Threshold from prompt
            self.metaspace_power_on = False
            
        if self.metaspace_power_on:
            power_metaspace = (self.current_target ** 2) * self.impedance
        else:
            power_metaspace = 0
            
        delta_temp_metaspace = (power_metaspace * self.dt / self.heat_capacity) - (self.metaspace_temp - 37.0) * self.cooling_factor * self.dt
        self.metaspace_temp += delta_temp_metaspace
        
        self.metaspace_temps.append(self.metaspace_temp)
        
        # --- 4. Update Plots ---
        ax1.clear()
        ax2.clear()
        
        # Top Plot: Impedance
        ax1.plot(self.time_data, self.impedance_data, color='blue', label='Coil Impedance')
        ax1.set_ylabel('Impedance (Ω)')
        ax1.set_title('MetaSpace InductiveGuard™: Vectorial SPS Analysis')
        ax1.grid(True, alpha=0.3)
        ax1.legend(loc='upper left')
        
        # Annotate Fracture
        if self.frame_count > self.fracture_frame:
            ax1.axvline(x=self.fracture_frame * self.dt, color='orange', linestyle='--', alpha=0.5)
            ax1.text(self.fracture_frame * self.dt + 0.5, 10, 'MICRO-FRACTURE\nDETECTED', color='orange', fontweight='bold')
        
        # Bottom Plot: Temperature
        ax2.plot(self.time_data, self.legacy_temps, color='red', linewidth=2, label='Legacy System (Thermistor)')
        ax2.plot(self.time_data, self.metaspace_temps, color='green', linewidth=2, label='MetaSpace (Predictive)')
        
        ax2.axhline(y=43.0, color='red', linestyle=':', alpha=0.5, label='Tissue Damage Threshold (43°C)')
        ax2.set_ylabel('Tissue Temp (°C)')
        ax2.set_xlabel('Time (s)')
        ax2.grid(True, alpha=0.3)
        ax2.legend(loc='upper left')
        
        # Annotations
        if not self.metaspace_power_on:
            ax2.text(self.time_data[-1] - 3, 37.5, 'SAFE SHUTDOWN\n(Pre-heating)', color='green', fontweight='bold')
            
        if self.legacy_temp > 43.0:
            ax2.text(self.time_data[-1] - 3, 44, 'TISSUE DAMAGE\nRISK', color='red', fontweight='bold')

# --- Setup Animation ---
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(10, 8), sharex=True)
plt.subplots_adjust(hspace=0.3)

monitor = InductiveMonitor()

def animate(i):
    monitor.update(i)

ani = animation.FuncAnimation(fig, animate, interval=50)
plt.show()
